﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.ServiceModel;

namespace VA.PPMS.CRM.Plugins
{
    public class LeieExclusionCreate : IPlugin
    {
        private const string PluginName = "LeieExclusionCreate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            ITracingService tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "ppms_leieexclusion")
                    return;

                tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    OptionSetValue action = entity.GetAttributeValue<OptionSetValue>("ppms_action");
                    string targetNpi = entity.GetAttributeValue<string>("ppms_npi");

                    if (String.IsNullOrEmpty(targetNpi))
                    {
                        tracingService.Trace("Check for matching provider by composite");
                        // Check for exclusion by demographics
                        EntityCollection result = GetMatchingExclusionsByComposite(service, entity);
                        if (result != null && result.Entities.Count > 0)
                        {
                            tracingService.Trace("Providers found");
                            foreach (var item in result.Entities)
                            {
                                tracingService.Trace("Composite deactivate request");
                                // Update LEIE properties on provider
                                UpdateLeieProperty(service, item, action.Value == (int)PpmsHelper.LeieAction.Exclude ? true : false);
                                // Update provider status
                                PpmsHelper.SetProviderState(service, item, action.Value);
                                CreateBatchDetail(service, entity, item);
                            }
                        }
                    }
                    else
                    {
                        // Check NPI value
                        tracingService.Trace("Retrieve provider by NPI");
                        var npis = GetMatchingNpis(service, entity.GetAttributeValue<string>("ppms_npi"), action.Value);
                        if (npis != null && npis.Entities.Count > 0)
                        {
                            tracingService.Trace("NPIs found");
                            // Deactivate providers
                            foreach (var npi in npis.Entities)
                            {
                                var provider = PpmsHelper.GetProvider(service, npi);
                                tracingService.Trace("Retrieve provider {0}", npi.GetAttributeValue<string>("ppms_name"));
                                // Update LEIE properties on provider
                                UpdateLeieProperty(service, provider, action.Value == (int)PpmsHelper.LeieAction.Exclude ? true : false);
                                // Update provider status
                                PpmsHelper.SetProviderState(service, provider, action.Value);
                                CreateBatchDetail(service, entity, provider);
                            }
                        }
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            tracingService.Trace("Done");
        }

        private void CreateBatchDetail(IOrganizationService service, Entity exclusion, Entity provider)
        {
            // Check for required parameters
            if (exclusion == null || provider == null) return;

            // Check for batch record
            //var batch = exclusion.GetAttributeValue<Entity>("ppms_batch");
            var batch = exclusion.GetAttributeValue<EntityReference>("ppms_batch");
            if (batch != null)
            {
                // Add detail response to batch detail
                var detail = new Entity("ppms_batchdetail");
                detail.Attributes.Add("ppms_name", provider.GetAttributeValue<string>("name"));
                detail.Attributes.Add("ppms_isvalid", false);
                detail.Attributes.Add("ppms_provider", provider.ToEntityReference());
                detail.Attributes.Add("ppms_transactiontype", new OptionSetValue((int)PpmsHelper.Batch_TransType.LeieExclusion));
                detail.Attributes.Add("ppms_batch", batch);
                var detailid = service.Create(detail);

                // Add detail to batch
                var result = new Entity("ppms_batchdetailresult");
                result.Attributes.Add("ppms_name", "LEIE Exclusion");
                result.Attributes.Add("ppms_isvalid", false);
                result.Attributes.Add("ppms_entitytype", "Provider");
                result.Attributes.Add("ppms_result", "Affected by LEIE Exclusion rules");
                result.Attributes.Add("ppms_message", exclusion.GetAttributeValue<string>("ppms_description"));
                result.Attributes.Add("ppms_batchdetail", new EntityReference("ppms_batchdetail", detailid));
                service.Create(result);
            }
        }

        private EntityCollection GetMatchingNpis(IOrganizationService service, string npi, int action)
        {
            int statecode = 0;

            if (String.IsNullOrEmpty(npi))
                return null;

            statecode = PpmsHelper.MapLeieActionToProviderState(action);

            FilterExpression filter = new FilterExpression();
            filter.AddCondition("ppms_provideridentifier", ConditionOperator.Equal, npi);
            filter.AddCondition("statecode", ConditionOperator.Equal, statecode);

            QueryExpression query = new QueryExpression("ppms_provideridentifier");
            query.ColumnSet.AddColumns("ppms_provideridentifierid", "ppms_provideridentifier", "ppms_providerid");
            query.Criteria.AddFilter(filter);

            return service.RetrieveMultiple(query);
        }

        // Retrieve matching provider records based on a composite of LEIE Exclusion fields
        public EntityCollection GetMatchingExclusionsByComposite(IOrganizationService service, Entity leie)
        {
            EntityCollection results = new EntityCollection();
            string addressComposite = PpmsHelper.GetAddressComposite(leie);
            // map LEIE action to account state
            int accountState = MapAccountState(leie);

            //define query
            QueryExpression query = new QueryExpression("account");
            query.ColumnSet.AddColumns("accountid", "name", "statecode");

            // retrieve last name first composite
            string name1 = String.Format("{0} {1} {2}%",
                leie.GetAttributeValue<string>("ppms_lastname"),
                leie.GetAttributeValue<string>("ppms_firstname"),
                leie.GetAttributeValue<string>("ppms_middlename")
            );

            // retrieve full name composite
            string name2 = String.Format("{0} {1} {2}%",
                leie.GetAttributeValue<string>("ppms_firstname"),
                leie.GetAttributeValue<string>("ppms_middlename"),
                leie.GetAttributeValue<string>("ppms_lastname")
            );

            // add name filter
            FilterExpression filter = new FilterExpression(LogicalOperator.Or);
            filter.AddCondition("name", ConditionOperator.Like, name1.Trim());
            filter.AddCondition("name", ConditionOperator.Like, name2.Trim().Replace("  ", " "));
            query.Criteria.AddFilter(filter);

            // add address and state filter
            FilterExpression filter2 = new FilterExpression();
            filter2.AddCondition("address1_composite", ConditionOperator.Like, addressComposite);
            filter2.AddCondition("statecode", ConditionOperator.Equal, accountState);
            query.Criteria.AddFilter(filter2);

            // retrieve results
            return service.RetrieveMultiple(query);
        }

        private int MapAccountState(Entity leie)
        {
            int statecode = 0;

            if (leie == null)
                return statecode;

            // get LEIE option set value
            var action = leie.GetAttributeValue<OptionSetValue>("ppms_action");
            if (action == null)
                return statecode;

            // determine action based on current state
            statecode = PpmsHelper.MapLeieActionToProviderState(action.Value);

            return statecode;
        }

        private void UpdateLeieProperty(IOrganizationService service, Entity provider, bool stateValue)
        {
            provider.Attributes["ppms_externalleie"] =  stateValue;
            provider.Attributes["ppms_externalleiecheckdate"] = DateTime.Now;
            service.Update(provider);
        }
    }
}
